--========================================================--
-- Unseasonal Weather - Radio Awareness Thoughts
-- Subtle player thoughts to hint radio usage.
--========================================================--

if not UW then UW = {} end
if not UW.RadioAwareness then UW.RadioAwareness = {} end

local RA = UW.RadioAwareness

------------------------------------------------------------
-- CONFIG
------------------------------------------------------------

RA.EVENTS_PER_THOUGHT = 7   -- ~1 por semana
RA.MAX_THOUGHTS       = 4

RA.Thoughts = {
    "UW.Radio.Awareness.01",
    "UW.Radio.Awareness.02",
    "UW.Radio.Awareness.03",
    "UW.Radio.Awareness.04",
}

------------------------------------------------------------
-- MODDATA HELPERS (por jugador)
------------------------------------------------------------

local function getPlayerData(player)
    if not player then return nil end
    local md = player:getModData()
    md.UW_RadioAwareness = md.UW_RadioAwareness or {
        eventsExperienced = 0,
        thoughtsShown     = {},
        radioUsed         = false,
    }
    return md.UW_RadioAwareness
end

local function allThoughtsShown(data)
    return #data.thoughtsShown >= RA.MAX_THOUGHTS
end

local function getNextThought(data)
    for _, text in ipairs(RA.Thoughts) do
        local used = false
        for _, shown in ipairs(data.thoughtsShown) do
            if shown == text then
                used = true
                break
            end
        end
        if not used then
            return text
        end
    end
    return nil
end

------------------------------------------------------------
-- PUBLIC API
------------------------------------------------------------

-- Call this when a *relevant* climate event finishes or starts
function RA.OnClimateEventExperienced()
    local player = getPlayer()
    if not player then return end

    local data = getPlayerData(player)
    if not data or data.radioUsed or allThoughtsShown(data) then
        return
    end

    data.eventsExperienced = data.eventsExperienced + 1

    -- Trigger only every X events
    if data.eventsExperienced % RA.EVENTS_PER_THOUGHT ~= 0 then
        return
    end

    -- Only outdoors
    if player.isInside and player:isInside() then return end

    local thoughtKey = getNextThought(data)
    if not thoughtKey then return end

    local text = getText(thoughtKey)
    if not text or text == "" then return end

    player:Say(text)
    table.insert(data.thoughtsShown, thoughtKey)
end

-- Call this ONCE when player successfully uses any radio
function RA.OnRadioUsed()
    local player = getPlayer()
    if not player then return end

    local data = getPlayerData(player)
    if not data then return end

    data.radioUsed = true
end

------------------------------------------------------------
-- OPTIONAL AUTO-DETECTION (SAFE)
------------------------------------------------------------

-- If you already track radio usage elsewhere, you can REMOVE this
local function onDeviceText(guid, codes, x, y, z, text, device)
    if not device or not device.getDeviceData then return end
    local d = device:getDeviceData()
    if d and d:getIsTurnedOn() then
        RA.OnRadioUsed()
    end
end

Events.OnDeviceText.Add(onDeviceText)
